import json
import networkx as nx

# Load JSON data from file
try:
    with open("friends7.json", "r") as f:
        data = json.load(f)
    print("Successfully loaded friends7.json")
except FileNotFoundError:
    print("Error: friends7.json not found. Please ensure the file exists in the working directory.")
    exit(1)
except json.JSONDecodeError:
    print("Error: Invalid JSON format in friends7.json.")
    exit(1)

# Create an undirected graph
G = nx.Graph()

# Add edges from the JSON data (mutual connections)
for person, friends in data.items():
    for friend in friends:
        G.add_edge(person, friend)

# Find connected components
components = list(nx.connected_components(G))
print(f"Number of connected components: {len(components)}")

# If there are multiple components, connect them to ensure no islanded trees
if len(components) > 1:
    # Select the largest component as the base
    largest_component = max(components, key=len)
    largest_nodes = set(largest_component)
    
    # Connect each smaller component to the largest component
    for component in components:
        if component != largest_component:
            # Pick one node from the smaller component and one from the largest
            node_small = next(iter(component))
            node_large = next(iter(largest_component))
            G.add_edge(node_small, node_large)
            print(f"Added edge between {node_small} and {node_large} to connect components")

# Verify the graph is now connected
if nx.is_connected(G):
    print("Graph is now fully connected with no islanded trees.")
else:
    print("Graph is still disconnected; further connections needed.")

# Generate DOT representation
dot_lines = ["graph G {"]
dot_lines.append("  node [shape=circle];")
for node in G.nodes():
    dot_lines.append(f'  "{node}" [label="{node}"];')
for edge in G.edges():
    dot_lines.append(f'  "{edge[0]}" -- "{edge[1]}";')
dot_lines.append("}")

# Write to a .dot file
dot_content = "\n".join(dot_lines)
with open("friends_network.dot", "w") as f:
    f.write(dot_content)

print("DOT file 'friends_network.dot' has been generated.")