import json
import networkx as nx

# Load JSON data from file
try:
    with open("friends7.json", "r") as f:
        data = json.load(f)
    print("Successfully loaded friends7.json")
except FileNotFoundError:
    print("Error: friends7.json not found. Please ensure the file exists in the working directory.")
    exit(1)
except json.JSONDecodeError:
    print("Error: Invalid JSON format in friends7.json.")
    exit(1)

# Create an undirected graph
G = nx.Graph()

# Add edges from the JSON data (mutual connections)
for person, friends in data.items():
    for friend in friends:
        G.add_edge(person, friend)

# Find connected components
components = list(nx.connected_components(G))
print(f"Number of connected components: {len(components)}")

# Print component details for transparency
for i, component in enumerate(components, 1):
    print(f"Component {i} (size {len(component)}): {component}")

# Select the largest component
largest_component = max(components, key=len)
print(f"\nLargest component size: {len(largest_component)} nodes")
print(f"Largest component nodes: {largest_component}")

# Create a subgraph with only the largest component
largest_G = G.subgraph(largest_component).copy()

# Verify the subgraph is connected
if nx.is_connected(largest_G):
    print("Largest component is fully connected with no islanded trees.")
else:
    print("Error: Largest component is not connected.")

# Check if Katteyes is in the largest component
if "Katteyes" in largest_G.nodes():
    print("Katteyes is in the largest component, connected to:", list(largest_G.neighbors("Katteyes")))
else:
    print("Katteyes is not in the largest component.")

# Generate DOT representation for the largest component
dot_lines = ["graph G {"]
dot_lines.append("  rankdir=LR;")  # Horizontal layout for better visualization
dot_lines.append("  node [shape=circle, style=filled, fillcolor=lightblue];")
dot_lines.append('  "Katteyes" [fillcolor=pink, style=filled];')  # Highlight Katteyes if present
for node in largest_G.nodes():
    if node != "Katteyes":  # Avoid duplicating Katteyes
        dot_lines.append(f'  "{node}" [label="{node}"];')
for edge in largest_G.edges():
    dot_lines.append(f'  "{edge[0]}" -- "{edge[1]}";')
dot_lines.append("}")

# Write to a .dot file
dot_content = "\n".join(dot_lines)
with open("friends_network_largest.dot", "w") as f:
    f.write(dot_content)

print("DOT file 'friends_network_largest.dot' has been generated for the largest component.")