import json
import networkx as nx

# Load JSON data from file
try:
    with open("friends7.json", "r") as f:
        data = json.load(f)
    print("Successfully loaded friends7.json")
except FileNotFoundError:
    print("Error: friends7.json not found. Please ensure the file exists in the working directory.")
    exit(1)
except json.JSONDecodeError:
    print("Error: Invalid JSON format in friends7.json.")
    exit(1)

# Create an undirected graph
G = nx.Graph()

# Add edges from the JSON data
for person, friends in data.items():
    for friend in friends:
        G.add_edge(person, friend)

# Add new authentic connections from internet search
new_connections = [
    ("Caitlin Clark", "Kelsey Mitchell"),  # WNBA teammates
    ("MrBeast", "Elon Musk"),  # Influencer collaboration
    ("Charli D'Amelio", "Kim Kardashian"),  # Social media collaboration
    ("Kim Kardashian", "Elon Musk")  # Public figure interaction
]
G.add_edges_from(new_connections)
print("Added new connections:", new_connections)

# Find connected components
components = list(nx.connected_components(G))
print(f"Number of connected components after adding edges: {len(components)}")

# Print component details
for i, component in enumerate(components, 1):
    print(f"Component {i} (size {len(component)}): {component}")

# Check if Katteyes is in the largest component
largest_component = max(components, key=len)
print(f"\nLargest component size: {len(largest_component)} nodes")
print(f"Largest component nodes: {largest_component}")

if "Katteyes" in largest_component:
    print("Katteyes is in the largest component, connected to:", list(G.neighbors("Katteyes")))
else:
    print("Katteyes is not in the largest component.")

# Generate DOT representation for the entire graph
dot_lines = ["graph G {"]
dot_lines.append("  rankdir=LR;")  # Horizontal layout
dot_lines.append("  node [shape=circle, style=filled, fillcolor=lightblue];")
dot_lines.append('  "Katteyes" [fillcolor=pink, style=filled];')  # Highlight Katteyes
for node in G.nodes():
    if node != "Katteyes":
        dot_lines.append(f'  "{node}" [label="{node}"];')
for edge in G.edges():
    dot_lines.append(f'  "{edge[0]}" -- "{edge[1]}";')

# Add subgraphs for visual separation if multiple components remain
for i, component in enumerate(components, 1):
    dot_lines.append(f"  subgraph cluster_{i} {{")
    dot_lines.append(f'    label="Component {i}";')
    dot_lines.append(f"    color=blue;")
    for node in component:
        dot_lines.append(f'    "{node}";')
    dot_lines.append("  }")
dot_lines.append("}")

# Write to a .dot file
dot_content = "\n".join(dot_lines)
with open("friends_network_full.dot", "w") as f:
    f.write(dot_content)

print("DOT file 'friends_network_full.dot' has been generated for the entire graph.")