import json
import networkx as nx
import uuid

# Load JSON data from file
try:
    with open("friends7.json", "r") as f:
        data = json.load(f)
    print("Successfully loaded friends7.json")
except FileNotFoundError:
    print("Error: friends7.json not found. Please ensure the file exists in the working directory.")
    exit(1)
except json.JSONDecodeError:
    print("Error: Invalid JSON format in friends7.json.")
    exit(1)

# Create an undirected graph
G = nx.Graph()

# Add edges from the JSON data
for person, friends in data.items():
    for friend in friends:
        G.add_edge(person, friend)

# Add new authentic connections
new_connections = [
    # Previous connections
    ("Caitlin Clark", "Kelsey Mitchell"),  # WNBA teammates
    ("MrBeast", "Elon Musk"),  # Influencer collaboration
    ("Charli D'Amelio", "Kim Kardashian"),  # Social media collaboration
    ("Kim Kardashian", "Elon Musk"),  # Public figure interaction
    ("Caitlin Clark", "Sabrina Ionescu"),  # WNBA connection
    ("Aaron Judge", "Alex Rodriguez"),  # Yankees mentorship
    ("Adin Ross", "MrBeast"),  # Streaming collaboration
    ("Alex Morgan", "Serena Williams"),  # Athlete endorsement/event
    ("Hasan Piker", "Joe Rogan"),  # Podcasting community
    # New connections for listed figures
    ("Jennifer Lawrence", "Robert Downey Jr."),  # Marvel franchise
    ("Orlando Bloom", "Katy Perry"),  # Engaged couple
    ("Harrison Ford", "Mark Hamill"),  # Star Wars co-stars
    ("Willie Nelson", "Bruce Springsteen"),  # Farm Aid collaboration
    ("J.K. Rowling", "Emma Watson"),  # Harry Potter franchise
    ("Emma Watson", "Eddie Redmayne"),  # Harry Potter/Fantastic Beasts
    ("Tyra Banks", "Kim Kardashian"),  # Reality TV/fashion
    ("Jackie Chan", "Chris Tucker"),  # Rush Hour co-stars
    ("Mick Jagger", "Elton John"),  # Music events
    ("Michael B. Jordan", "Ryan Coogler"),  # Creed/Black Panther
    ("Mark Zuckerberg", "Elon Musk"),  # Tech mogul interactions
    ("Barack Obama", "Beyoncé"),  # Political events/endorsements
    ("Arnold Schwarzenegger", "Sylvester Stallone"),  # Expendables co-stars
    ("Conan O’Brien", "Will Ferrell"),  # Late-night show appearances
    ("Jon Stewart", "John Oliver"),  # The Daily Show
    ("Justin Trudeau", "Barack Obama"),  # Diplomatic friendship
    ("Serena Williams", "Venus Williams"),  # Sisters/tennis partners
    ("Stephen Hawking", "Neil deGrasse Tyson")  # Science communication
]
G.add_edges_from(new_connections)
print("Added new connections:", new_connections)

# Find connected components
components = list(nx.connected_components(G))
print(f"Number of connected components after adding edges: {len(components)}")

# Print component details
for i, component in enumerate(components, 1):
    print(f"Component {i} (size {len(component)}): {component}")

# Check if the graph is fully connected
if len(components) == 1:
    print("Graph is now a single connected tree with no islanded trees.")
else:
    print(f"Graph still has {len(components)} components. Some islanders remain as expected.")

# Check Katteyes' status
if "Katteyes" in G.nodes():
    for component in components:
        if "Katteyes" in component:
            print(f"Katteyes is in a component of size {len(component)}, connected to: {list(G.neighbors('Katteyes'))}")
            break
else:
    print("Katteyes is not in the graph.")

# Generate DOT representation for the entire graph
dot_lines = ["graph G {"]
dot_lines.append("  rankdir=LR;")  # Horizontal layout
dot_lines.append("  node [shape=circle, style=filled, fillcolor=lightblue];")
for node in G.nodes():
    dot_lines.append(f'  "{node}" [label="{node}"];')
for edge in G.edges():
    dot_lines.append(f'  "{edge[0]}" -- "{edge[1]}";')

# Add subgraphs for visual separation if multiple components remain
if len(components) > 1:
    for i, component in enumerate(components, 1):
        dot_lines.append(f"  subgraph cluster_{i} {{")
        dot_lines.append(f'    label="Component {i}";')
        dot_lines.append(f"    color=blue;")
        for node in component:
            dot_lines.append(f'    "{node}";')
        dot_lines.append("  }")
dot_lines.append("}")

# Write to a .dot file
dot_content = "\n".join(dot_lines)
with open("friends_network_full.dot", "w") as f:
    f.write(dot_content)

print("DOT file 'friends_network_full.dot' has been generated for the entire graph.")