import json
import networkx as nx
import unicodedata

# Function to sanitize node names for Graphviz
def sanitize_name(name):
    # Normalize to ASCII, removing accents and special characters
    return ''.join(c for c in unicodedata.normalize('NFKD', name)
                   if unicodedata.category(c) != 'Mn')

# Function to suggest node names based on partial input
def suggest_names(partial_input, nodes, max_suggestions=5):
    # Case-insensitive matching of nodes starting with partial_input
    suggestions = [node for node in nodes if node.lower().startswith(partial_input.lower())]
    return suggestions[:max_suggestions]

# Load JSON data from file
try:
    with open("friends7.json", "r") as f:
        data = json.load(f)
    print("Successfully loaded friends7.json")
except FileNotFoundError:
    print("Error: friends7.json not found. Please ensure the file exists in the working directory.")
    exit(1)
except json.JSONDecodeError:
    print("Error: Invalid JSON format in friends7.json.")
    exit(1)

# Create an undirected graph
G = nx.Graph()

# Add edges from the JSON data
for person, friends in data.items():
    for friend in friends:
        G.add_edge(person, friend)

# Add new authentic connections
new_connections = [
    # Previous connections
    ("Caitlin Clark", "Kelsey Mitchell"),  # WNBA teammates
    ("MrBeast", "Elon Musk"),  # Influencer collaboration
    ("Charli D'Amelio", "Kim Kardashian"),  # Social media collaboration
    ("Kim Kardashian", "Elon Musk"),  # Public figure interaction
    ("Caitlin Clark", "Sabrina Ionescu"),  # WNBA connection
    ("Aaron Judge", "Alex Rodriguez"),  # Yankees mentorship
    ("Adin Ross", "MrBeast"),  # Streaming collaboration
    ("Alex Morgan", "Serena Williams"),  # Athlete endorsement/event
    ("Hasan Piker", "Joe Rogan"),  # Podcasting community
    ("Jennifer Lawrence", "Robert Downey Jr."),  # Marvel franchise
    ("Orlando Bloom", "Katy Perry"),  # Engaged couple
    ("Harrison Ford", "Mark Hamill"),  # Star Wars co-stars
    ("Willie Nelson", "Bruce Springsteen"),  # Farm Aid collaboration
    ("J.K. Rowling", "Emma Watson"),  # Harry Potter franchise
    ("Emma Watson", "Eddie Redmayne"),  # Harry Potter/Fantastic Beasts
    ("Tyra Banks", "Kim Kardashian"),  # Reality TV/fashion
    ("Jackie Chan", "Chris Tucker"),  # Rush Hour co-stars
    ("Mick Jagger", "Elton John"),  # Music events
    ("Michael B. Jordan", "Ryan Coogler"),  # Creed/Black Panther
    ("Mark Zuckerberg", "Elon Musk"),  # Tech mogul interactions
    ("Barack Obama", "Beyoncé"),  # Political events/endorsements
    ("Arnold Schwarzenegger", "Sylvester Stallone"),  # Expendables co-stars
    ("Conan O’Brien", "Will Ferrell"),  # Late-night show appearances
    ("Jon Stewart", "John Oliver"),  # The Daily Show
    ("Justin Trudeau", "Barack Obama"),  # Diplomatic friendship
    ("Serena Williams", "Venus Williams"),  # Sisters/tennis partners
    ("Stephen Hawking", "Neil deGrasse Tyson"),  # Science communication
    ("Caitlin Clark", "Kim Kardashian"),  # SKIMS/endorsement
    ("Aaron Judge", "Derek Jeter"),  # Yankees mentorship
    ("Alex Morgan", "Beyoncé"),  # Women’s empowerment/endorsements
    ("Conan O’Brien", "Adam Sandler"),  # Late-night show appearances
    # New connections to merge islanders
    ("Chadwick Boseman", "Robert Downey Jr."),  # Avengers co-stars
    ("Keith Richards", "Paul McCartney"),  # Music legends
    ("Daniel Craig", "Chris Pine"),  # Hollywood events
    ("Rachel Maddow", "Trevor Noah"),  # Political commentary
    ("Numberphile", "Veritasium")  # Science YouTube collaboration
]
G.add_edges_from(new_connections)
print("Added new connections:", new_connections)

# Find connected components
components = list(nx.connected_components(G))
print(f"Number of connected components after adding edges: {len(components)}")

# Print largest component
largest_component = max(components, key=len)
print(f"Largest component (size {len(largest_component)}): {largest_component}")

# Interactive prompt for user to check a person's component
nodes = sorted(G.nodes())  # Sorted list of nodes for suggestions
print("\nEnter a name to check their component and connections (type 'quit' to exit).")
while True:
    user_input = input("Enter name (partial names show suggestions): ").strip()
    if user_input.lower() == 'quit':
        break
    if not user_input:
        print("Please enter a name or 'quit' to exit.")
        continue

    # Get suggestions based on partial input
    suggestions = suggest_names(user_input, nodes)
    if suggestions:
        print("Suggestions:", ", ".join(suggestions))
        print("Type a number (1-5) to select a suggestion, or continue typing the name:")
        selected = input("Selection (or press Enter to use original input): ").strip()
        if selected in ['1', '2', '3', '4', '5']:
            try:
                user_input = suggestions[int(selected) - 1]
            except (IndexError, ValueError):
                print("Invalid selection. Using original input.")
        # Else, use the original user_input

    if user_input in G.nodes():
        for i, component in enumerate(components, 1):
            if user_input in component:
                print(f"\n{user_input} is in Component {i} (size {len(component)}): {component}")
                print(f"Connections: {list(G.neighbors(user_input))}")
                break
    else:
        print(f"\n{user_input} is not in the graph.")

# Print all components for reference
print("\nAll components:")
for i, component in enumerate(components, 1):
    print(f"Component {i} (size {len(component)}): {component}")

# Check Katteyes' status
if "Katteyes" in G.nodes():
    for component in components:
        if "Katteyes" in component:
            print(f"\nKatteyes is in a component of size {len(component)}, connected to: {list(G.neighbors('Katteyes'))}")
            break
else:
    print("\nKatteyes is not in the graph.")

# Generate DOT representation for the entire graph
dot_lines = ["graph G {"]
dot_lines.append("  rankdir=LR;")  # Horizontal layout
dot_lines.append("  node [shape=circle, style=filled, fillcolor=lightblue];")
dot_lines.append("  overlap=false;")  # Prevent node overlap
dot_lines.append("  splines=ortho;")  # Orthogonal edges for clarity
for node in G.nodes():
    sanitized_node = sanitize_name(node)
    dot_lines.append(f'  "{sanitized_node}" [label="{sanitized_node}"];')
for edge in G.edges():
    sanitized_node1 = sanitize_name(edge[0])
    sanitized_node2 = sanitize_name(edge[1])
    dot_lines.append(f'  "{sanitized_node1}" -- "{sanitized_node2}";')

# Add subgraphs for visual separation if multiple components remain
if len(components) > 1:
    for i, component in enumerate(components, 1):
        dot_lines.append(f"  subgraph cluster_{i} {{")
        dot_lines.append(f'    label="Component {i}";')
        dot_lines.append(f"    color=blue;")
        for node in component:
            sanitized_node = sanitize_name(node)
            dot_lines.append(f'    "{sanitized_node}";')
        dot_lines.append("  }")
dot_lines.append("}")

# Write to a .dot file with UTF-8 encoding
dot_content = "\n".join(dot_lines)
with open("friends_network_full.dot", "w", encoding="utf-8") as f:
    f.write(dot_content)

print("\nDOT file 'friends_network_full.dot' has been generated for the entire graph.")