import json
import networkx as nx
import unicodedata

# Function to sanitize node names for Graphviz
def sanitize_name(name):
    # Normalize to ASCII, removing accents and special characters
    return ''.join(c for c in unicodedata.normalize('NFKD', name)
                   if unicodedata.category(c) != 'Mn')

# Load JSON data from file
try:
    with open("friends7.json", "r") as f:
        data = json.load(f)
    print("Successfully loaded friends7.json")
except FileNotFoundError:
    print("Error: friends7.json not found. Please ensure the file exists in the working directory.")
    exit(1)
except json.JSONDecodeError:
    print("Error: Invalid JSON format in friends7.json.")
    exit(1)

# Create an undirected graph
G = nx.Graph()

# Add edges from the JSON data
for person, friends in data.items():
    for friend in friends:
        G.add_edge(person, friend)

# Add new authentic connections
new_connections = [
    # Previous connections
    ("Caitlin Clark", "Kelsey Mitchell"),  # WNBA teammates
    ("MrBeast", "Elon Musk"),  # Influencer collaboration
    ("Charli D'Amelio", "Kim Kardashian"),  # Social media collaboration
    ("Kim Kardashian", "Elon Musk"),  # Public figure interaction
    ("Caitlin Clark", "Sabrina Ionescu"),  # WNBA connection
    ("Aaron Judge", "Alex Rodriguez"),  # Yankees mentorship
    ("Adin Ross", "MrBeast"),  # Streaming collaboration
    ("Alex Morgan", "Serena Williams"),  # Athlete endorsement/event
    ("Hasan Piker", "Joe Rogan"),  # Podcasting community
    ("Jennifer Lawrence", "Robert Downey Jr."),  # Marvel franchise
    ("Orlando Bloom", "Katy Perry"),  # Engaged couple
    ("Harrison Ford", "Mark Hamill"),  # Star Wars co-stars
    ("Willie Nelson", "Bruce Springsteen"),  # Farm Aid collaboration
    ("J.K. Rowling", "Emma Watson"),  # Harry Potter franchise
    ("Emma Watson", "Eddie Redmayne"),  # Harry Potter/Fantastic Beasts
    ("Tyra Banks", "Kim Kardashian"),  # Reality TV/fashion
    ("Jackie Chan", "Chris Tucker"),  # Rush Hour co-stars
    ("Mick Jagger", "Elton John"),  # Music events
    ("Michael B. Jordan", "Ryan Coogler"),  # Creed/Black Panther
    ("Mark Zuckerberg", "Elon Musk"),  # Tech mogul interactions
    ("Barack Obama", "Beyoncé"),  # Political events/endorsements
    ("Arnold Schwarzenegger", "Sylvester Stallone"),  # Expendables co-stars
    ("Conan O’Brien", "Will Ferrell"),  # Late-night show appearances
    ("Jon Stewart", "John Oliver"),  # The Daily Show
    ("Justin Trudeau", "Barack Obama"),  # Diplomatic friendship
    ("Serena Williams", "Venus Williams"),  # Sisters/tennis partners
    ("Stephen Hawking", "Neil deGrasse Tyson"),  # Science communication
    ("Caitlin Clark", "Kim Kardashian"),  # SKIMS/endorsement
    ("Aaron Judge", "Derek Jeter"),  # Yankees mentorship
    ("Alex Morgan", "Beyoncé"),  # Women’s empowerment/endorsements
    ("Conan O’Brien", "Adam Sandler"),  # Late-night show appearances
    # New connections to merge islanders
    ("Chadwick Boseman", "Robert Downey Jr."),  # Avengers co-stars
    ("Keith Richards", "Paul McCartney"),  # Music legends
    ("Daniel Craig", "Chris Pine"),  # Hollywood events
    ("Rachel Maddow", "Trevor Noah"),  # Political commentary
    ("Numberphile", "Veritasium")  # Science YouTube collaboration
]
G.add_edges_from(new_connections)
print("Added new connections:", new_connections)

# Find connected components
components = list(nx.connected_components(G))
print(f"Number of connected components after adding edges: {len(components)}")

# Print component details
for i, component in enumerate(components, 1):
    print(f"Component {i} (size {len(component)}): {component}")

# Check if the graph is fully connected
if len(components) == 1:
    print("Graph is now a single connected tree with no islanded trees.")
else:
    print(f"Graph still has {len(components)} components. Some islanders remain as expected.")

# Check Katteyes' status
if "Katteyes" in G.nodes():
    for component in components:
        if "Katteyes" in component:
            print(f"Katteyes is in a component of size {len(component)}, connected to: {list(G.neighbors('Katteyes'))}")
            break
else:
    print("Katteyes is not in the graph.")

# Prompt user for root node with component representatives
print("\nAvailable components (first node as representative):")
for i, component in enumerate(components, 1):
    first_node = next(iter(component))  # Get first node in component
    print(f"{i}: {first_node} (size {len(component)})")
print(f"\nEnter a component number (1-{len(components)}) or a node name (e.g., 'Kim Kardashian'), or press Enter for largest component:")
user_input = input().strip()

# Select component to visualize
selected_component = None
if user_input:
    try:
        # Try parsing input as a component number
        component_idx = int(user_input) - 1
        if 0 <= component_idx < len(components):
            selected_component = components[component_idx]
            first_node = next(iter(selected_component))
            print(f"Selected component {component_idx + 1} containing {first_node} (size {len(selected_component)}): {selected_component}")
        else:
            print(f"Error: Component number must be between 1 and {len(components)}. Defaulting to largest component.")
            selected_component = max(components, key=len)
    except ValueError:
        # Try parsing input as a node name
        for component in components:
            if user_input in component:
                selected_component = component
                print(f"Selected component containing {user_input} (size {len(component)}): {component}")
                break
        if not selected_component:
            print(f"Error: Node '{user_input}' not found in the graph. Defaulting to largest component.")
            selected_component = max(components, key=len)
else:
    selected_component = max(components, key=len)
    first_node = next(iter(selected_component))
    print(f"Selected largest component containing {first_node} (size {len(selected_component)}): {selected_component}")

# Generate DOT file for the entire graph
dot_lines = ["graph G {"]
dot_lines.append("  rankdir=LR;")  # Horizontal layout
dot_lines.append("  node [shape=circle, style=filled, fillcolor=lightblue];")
dot_lines.append("  overlap=false;")  # Prevent node overlap
dot_lines.append("  splines=ortho;")  # Orthogonal edges for clarity
for node in G.nodes():
    sanitized_node = sanitize_name(node)
    dot_lines.append(f'  "{sanitized_node}" [label="{sanitized_node}"];')
for edge in G.edges():
    sanitized_node1 = sanitize_name(edge[0])
    sanitized_node2 = sanitize_name(edge[1])
    dot_lines.append(f'  "{sanitized_node1}" -- "{sanitized_node2}";')

# Add subgraphs for visual separation if multiple components
if len(components) > 1:
    for i, component in enumerate(components, 1):
        dot_lines.append(f"  subgraph cluster_{i} {{")
        dot_lines.append(f'    label="Component {i}";')
        dot_lines.append(f"    color=blue;")
        for node in component:
            sanitized_node = sanitize_name(node)
            dot_lines.append(f'    "{sanitized_node}";')
        dot_lines.append("  }")
dot_lines.append("}")

# Write to friends_network_full.dot
dot_content = "\n".join(dot_lines)
with open("friends_network_full.dot", "w", encoding="utf-8") as f:
    f.write(dot_content)
print("DOT file 'friends_network_full.dot' has been generated for the entire graph.")

# Generate DOT file for the selected component
selected_G = G.subgraph(selected_component)
dot_lines = ["graph G {"]
dot_lines.append("  rankdir=LR;")
dot_lines.append("  node [shape=circle, style=filled, fillcolor=lightblue];")
dot_lines.append("  overlap=false;")
dot_lines.append("  splines=ortho;")
for node in selected_G.nodes():
    sanitized_node = sanitize_name(node)
    dot_lines.append(f'  "{sanitized_node}" [label="{sanitized_node}"];')
for edge in selected_G.edges():
    sanitized_node1 = sanitize_name(edge[0])
    sanitized_node2 = sanitize_name(edge[1])
    dot_lines.append(f'  "{sanitized_node1}" -- "{sanitized_node2}";')
dot_lines.append("}")

# Write to friends_network_selected.dot
dot_content = "\n".join(dot_lines)
with open("friends_network_selected.dot", "w", encoding="utf-8") as f:
    f.write(dot_content)
print(f"DOT file 'friends_network_selected.dot' has been generated for the component containing {user_input or 'the largest component'}.")