import json
from collections import defaultdict

# Load JSON data from a file
def load_json(file_path):
    try:
        with open(file_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: {file_path} not found.")
        return {}

# Combine connections from all JSON files
def combine_connections(json_files):
    connections = defaultdict(set)
    
    for file in json_files:
        data = load_json(file)
        for person, friends in data.items():
            for friend in friends:
                # Add undirected edge (both directions)
                connections[person].add(friend)
                connections[friend].add(person)
    
    # Convert to a set of unique edges
    edges = set()
    for person, friends in connections.items():
        for friend in sorted(friends):  # Sort to ensure consistent edge order
            if person < friend:  # Avoid duplicates (e.g., A-B same as B-A)
                edges.add((person, friend))
            elif friend < person:
                edges.add((friend, person))
    
    return edges

# Generate DOT file for Graphviz
def generate_dot_file(edges, output_file, max_edges=500):
    with open(output_file, 'w') as f:
        f.write("graph friendships {\n")
        f.write("    rankdir=LR;\n")  # Left-to-right layout
        f.write("    node [shape=circle, style=filled, fillcolor=lightblue];\n")
        f.write("    edge [color=navy];\n")
        
        # Write edges, limiting to max_edges for readability
        for i, (person, friend) in enumerate(edges):
            if i >= max_edges:
                print(f"Warning: Limited output to {max_edges} edges for readability.")
                break
            f.write(f'    "{person}" -- "{friend}";\n')
        
        f.write("}\n")
    print(f"DOT file generated: {output_file}")

# Main execution
if __name__ == "__main__":
    json_files = [
        'friends1.json',
        'friends2.json',
        'friends3.json',
        'friends4.json',
        'friends5.json',
        'friends6.json'
    ]
    
    # Combine connections
    edges = combine_connections(json_files)
    
    # Generate DOT file
    generate_dot_file(edges, 'friendship_graph.dot')