import json
from collections import defaultdict

# Load JSON data
def load_json(file_path):
    try:
        with open(file_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: {file_path} not found.")
        return {}

# Combine connections from all JSON files
def combine_connections(json_files):
    connections = defaultdict(set)
    
    for file in json_files:
        data = load_json(file)
        for person, friends in data.items():
            for friend in friends:
                connections[person].add(friend)
                connections[friend].add(person)  # Ensure mutual connections
    
    return connections

# Generate DOT file for the complete graph
def generate_dot_file(connections, output_file, max_nodes=1000):
    with open(output_file, 'w') as f:
        f.write("graph friendships {\n")
        f.write("    rankdir=LR;\n")  # Left-to-right layout for clarity
        f.write("    node [shape=circle, style=filled, fillcolor=lightblue];\n")
        f.write("    edge [color=navy];\n")
        
        # Track nodes to limit visualization
        nodes_written = set()
        for person, friends in connections.items():
            if len(nodes_written) >= max_nodes:
                break
            nodes_written.add(person)
            for friend in friends:
                if friend in connections and len(nodes_written) < max_nodes:
                    nodes_written.add(friend)
                    f.write(f'    "{person}" -- "{friend}";\n')
        
        f.write("}\n")
    print(f"DOT file generated: {output_file}")

# Main execution
if __name__ == "__main__":
    json_files = [
        'friends1.json',
        'friends2.json',
        'friends3.json',
        'friends4.json',
        'friends5.json',
        'friends6.json'
    ]
    
    # Combine connections
    connections = combine_connections(json_files)
    
    # Generate DOT file
    generate_dot_file(connections, 'friendship_graph.dot', max_nodes=1000)