import json
import re
from collections import defaultdict

# Function to normalize names
def normalize_name(name):
    # Normalize: lowercase, remove accents, handle variations
    name = re.sub(r'[^\w\s]', '', name.lower().replace('é', 'e').replace('á', 'a').replace('ñ', 'n'))
    name_map = {
        'leo messi': 'lionel messi',
        'kylian mbappe': 'kylian mbappé',
        'beyonce': 'beyoncé',
        'amanda sc gorman': 'amanda gorman',
        'nikkietutorials': 'nikkie de jager',
        'the stokes twins': 'stokes twins',
        'kourtney kardashian barker': 'kourtney kardashian'
    }
    return name_map.get(name, name)

# Load JSON data
def load_json(file_path):
    try:
        with open(file_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: {file_path} not found.")
        return {}

# Combine connections from all JSON files
def combine_connections(json_files):
    connections = defaultdict(set)
    
    for file in json_files:
        data = load_json(file)
        for person, friends in data.items():
            norm_person = normalize_name(person)
            for friend in friends:
                norm_friend = normalize_name(friend)
                connections[norm_person].add(norm_friend)
                connections[norm_friend].add(norm_person)  # Ensure mutual connections
    
    return connections

# Generate DOT file for the full graph
def generate_dot_file(connections, output_file, max_nodes=2000):
    with open(output_file, 'w') as f:
        f.write("graph friendships {\n")
        f.write("    rankdir=LR;\n")  # Left-to-right layout for readability
        f.write("    node [shape=circle, style=filled, fillcolor=lightblue];\n")
        f.write("    edge [color=navy];\n")
        
        nodes_written = set()
        edges_written = set()
        node_count = 0
        
        for person, friends in connections.items():
            if node_count >= max_nodes:
                break
            nodes_written.add(person)
            for friend in friends:
                if friend in connections and (person, friend) not in edges_written and (friend, person) not in edges_written:
                    edges_written.add((person, friend))
                    f.write(f'    "{person}" -- "{friend}";\n')
            node_count += 1
        
        f.write("}\n")
    print(f"DOT file generated: {output_file}")

# Main execution
if __name__ == "__main__":
    json_files = [
        'friends1.json',
        'friends2.json',
        'friends3.json',
        'friends4.json',
        'friends5.json',
        'friends6.json'
    ]
    
    # Combine connections
    connections = combine_connections(json_files)
    
    # Generate DOT file for the full graph
    generate_dot_file(connections, 'friendship_graph.dot')