import json
import re
from collections import defaultdict
import networkx as nx

# Function to normalize names
def normalize_name(name):
    # Simple normalization: lowercase, remove accents, handle variations
    name = re.sub(r'[^\w\s]', '', name.lower().replace('é', 'e').replace('á', 'a').replace('ñ', 'n'))
    name_map = {
        'leo messi': 'lionel messi',
        'kylian mbappe': 'kylian mbappé',
        'beyonce': 'beyoncé',
        'amanda sc gorman': 'amanda gorman',
        'nikkietutorials': 'nikkie de jager'
    }
    return name_map.get(name, name)

# Load JSON data
def load_json(file_path):
    try:
        with open(file_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: {file_path} not found.")
        return {}

# Placeholder for fetching data from zchg.org
def fetch_zchg_data(url):
    # Placeholder: Replace with actual web scraping logic
    print(f"Fetching data from {url}...")
    return {}
    # Example implementation (uncomment and adjust if you have access):
    # import requests
    # from bs4 import BeautifulSoup
    # response = requests.get(url)
    # soup = BeautifulSoup(response.text, 'html.parser')
    # connections = {}
    # # Parse HTML to extract person -> friends mappings
    # for item in soup.find_all('div', class_='connection'):
    #     person = item.find('span', class_='person').text.strip()
    #     friends = [friend.text.strip() for friend in item.find_all('span', class_='friend')]
    #     connections[person] = friends
    # return connections

# Combine connections from all sources
def combine_connections(json_files, urls):
    connections = defaultdict(set)
    
    # Process JSON files
    for file in json_files:
        data = load_json(file)
        for person, friends in data.items():
            norm_person = normalize_name(person)
            for friend in friends:
                norm_friend = normalize_name(friend)
                connections[norm_person].add(norm_friend)
                connections[norm_friend].add(norm_person)  # Ensure mutual connections
    
    # Process URL data
    for url in urls:
        data = fetch_zchg_data(url)
        for person, friends in data.items():
            norm_person = normalize_name(person)
            for friend in friends:
                norm_friend = normalize_name(friend)
                connections[norm_person].add(norm_friend)
                connections[norm_friend].add(norm_person)
    
    return connections

# Generate a tree-like structure rooted at a specific node
def build_tree(connections, root, max_depth=2):
    G = nx.Graph()
    for person, friends in connections.items():
        for friend in friends:
            G.add_edge(person, friend)
    
    # Perform BFS to build a tree rooted at the specified node
    tree_edges = []
    visited = {root}
    queue = [(root, 0)]
    while queue:
        node, depth = queue.pop(0)
        if depth >= max_depth:
            continue
        for neighbor in connections[node]:
            if neighbor not in visited:
                visited.add(neighbor)
                tree_edges.append((node, neighbor))
                queue.append((neighbor, depth + 1))
    
    return tree_edges

# Generate DOT file
def generate_dot_file(connections, output_file, root=None, max_depth=2, max_nodes=50):
    with open(output_file, 'w') as f:
        f.write("graph friendships {\n")
        f.write("    rankdir=BT;\n")  # Bottom-to-top for tree-like display
        f.write("    node [shape=circle, style=filled, fillcolor=lightblue];\n")
        f.write("    edge [color=navy];\n")
        
        if root:
            # Generate a tree rooted at the specified node
            tree_edges = build_tree(connections, root, max_depth)
            nodes_written = set()
            for edge in tree_edges:
                node1, node2 = edge
                nodes_written.add(node1)
                nodes_written.add(node2)
                f.write(f'    "{node1}" -- "{node2}";\n')
                if len(nodes_written) >= max_nodes:
                    break
        else:
            # Generate full graph (limited for clarity)
            nodes_written = 0
            for person, friends in connections.items():
                if nodes_written >= max_nodes:
                    break
                for friend in friends:
                    if friend in connections:
                        f.write(f'    "{person}" -- "{friend}";\n')
                nodes_written += 1
        
        f.write("}\n")
    print(f"DOT file generated: {output_file}")

# Main execution
if __name__ == "__main__":
    json_files = [
        'friends1.json',
        'friends2.json',
        'friends3.json',
        'friends4.json',
        'friends5.json',
        'friends6.json'
    ]
    urls = ['https://zchg.org/t/six-degrees-of-separation/842/1']
    
    # Combine connections
    connections = combine_connections(json_files, urls)
    
    # Generate DOT file with a tree rooted at "Cristiano Ronaldo"
    generate_dot_file(connections, 'friendship_tree.dot', root='cristiano ronaldo', max_depth=2)