import json
from collections import defaultdict

# Load JSON data from a file
def load_json(file_path):
    try:
        with open(file_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: {file_path} not found.")
        return {}

# Combine connections from all JSON files
def combine_connections(json_files):
    connections = defaultdict(set)
    
    for file in json_files:
        data = load_json(file)
        for person, friends in data.items():
            for friend in friends:
                # Add undirected edge (both directions)
                connections[person].add(friend)
                connections[friend].add(person)
    
    # Convert to a set of unique edges
    edges = set()
    for person, friends in connections.items():
        for friend in sorted(friends):  # Sort to ensure consistent edge order
            if person < friend:  # Avoid duplicates (e.g., A-B same as B-A)
                edges.add((person, friend))
            elif friend < person:
                edges.add((friend, person))
    
    return edges, connections

# Generate DOT file for Graphviz
def generate_dot_file(edges, output_file):
    with open(output_file, 'w') as f:
        f.write("graph friendships {\n")
        f.write("    rankdir=LR;\n")  # Left-to-right layout
        f.write("    node [shape=circle, style=filled, fillcolor=lightblue];\n")
        f.write("    edge [color=navy];\n")
        
        # Write all edges
        for person, friend in edges:
            f.write(f'    "{person}" -- "{friend}";\n')
        
        f.write("}\n")
    print(f"DOT file generated: {output_file} with {len(edges)} edges")

# Optional: Filter by category (e.g., sports, musicians)
def filter_by_category(connections, category_nodes, output_file):
    edges = set()
    for person in category_nodes:
        if person in connections:
            for friend in connections[person]:
                if friend in category_nodes and person < friend:
                    edges.add((person, friend))
    
    with open(output_file, 'w') as f:
        f.write("graph filtered_friendships {\n")
        f.write("    rankdir=LR;\n")
        f.write("    node [shape=circle, style=filled, fillcolor=lightcoral];\n")
        f.write("    edge [color=navy];\n")
        for person, friend in edges:
            f.write(f'    "{person}" -- "{friend}";\n')
        f.write("}\n")
    print(f"Filtered DOT file generated: {output_file} with {len(edges)} edges")

# Optional: Cluster by JSON file
def generate_clustered_dot(json_files, output_file):
    with open(output_file, 'w') as f:
        f.write("graph friendships_clustered {\n")
        f.write("    rankdir=LR;\n")
        f.write("    node [shape=circle, style=filled];\n")
        f.write("    edge [color=navy];\n")
        
        for i, file in enumerate(json_files):
            data = load_json(file)
            f.write(f"    subgraph cluster_{i} {{\n")
            f.write(f"        label=\"{file}\";\n")
            f.write("        color=blue;\n")
            f.write("        fillcolor=lightgrey;\n")
            f.write("        style=filled;\n")
            nodes = set()
            edges = set()
            for person, friends in data.items():
                nodes.add(person)
                for friend in friends:
                    nodes.add(friend)
                    edge = tuple(sorted([person, friend]))
                    edges.add(edge)
            for node in nodes:
                f.write(f'        "{node}";\n')
            for person, friend in edges:
                f.write(f'        "{person}" -- "{friend}";\n')
            f.write("    }\n")
        
        f.write("}\n")
    print(f"Clustered DOT file generated: {output_file}")

# Main execution
if __name__ == "__main__":
    json_files = [
        'friends1.json',
        'friends2.json',
        'friends3.json',
        'friends4.json',
        'friends5.json',
        'friends6.json'
    ]
    
    # Combine connections
    edges, connections = combine_connections(json_files)
    
    # Generate full graph
    generate_dot_file(edges, 'friendship_graph_full.dot')
    
    # Example: Filter by sports figures (uncomment to use)
    # sports_nodes = [
    #     "Cristiano Ronaldo", "Leo Messi", "Neymar Jr.", "Kylian Mbappe",
    #     "David Beckham", "Virat Kohli"
    # ]
    # filter_by_category(connections, sports_nodes, 'friendship_sports.dot')
    
    # Example: Generate clustered graph (uncomment to use)
    # generate_clustered_dot(json_files, 'friendship_clustered.dot')