import json
import sys
from pathlib import Path

def load_json(file_path):
    """Load JSON file and return dictionary."""
    try:
        with open(file_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: File {file_path} not found.")
        sys.exit(1)
    except json.JSONDecodeError:
        print(f"Error: Invalid JSON format in {file_path}.")
        sys.exit(1)

def ensure_bidirectional(data):
    """Ensure bidirectional consistency in connections."""
    for person, connections in list(data.items()):
        for connection in connections:
            if connection not in data:
                data[connection] = []
            if person not in data[connection]:
                data[connection].append(person)
                data[connection].sort()  # Keep connections sorted

def inject_entries(existing_data, new_entries):
    """Inject new entries into existing data and ensure bidirectional consistency."""
    # Update or add new entries
    for person, connections in new_entries.items():
        existing_data[person] = sorted(list(set(existing_data.get(person, []) + connections)))
    
    # Ensure bidirectional consistency
    ensure_bidirectional(existing_data)
    
    # Sort the entire dictionary by keys
    sorted_data = dict(sorted(existing_data.items()))
    return sorted_data

def save_json(data, output_path):
    """Save dictionary to JSON file."""
    with open(output_path, 'w') as f:
        json.dump(data, f, indent=2, sort_keys=True)

def main():
    if len(sys.argv) != 3:
        print("Usage: python json_injector.py <input_json> <output_json>")
        sys.exit(1)
    
    input_path = sys.argv[1]
    output_path = sys.argv[2]
    
    # Example new entries (can be passed as input or modified)
    new_entries = {
  "Aaron Judge": ["Alex Rodriguez", "Derek Jeter", "Donald Trump", "Joe Rogan", "LeBron James", "Arnold Schwarzenegger", "Anthony Rizzo", "Giancarlo Stanton", "CC Sabathia", "Mariano Rivera", "Alex Bregman", "Jose Altuve", "Tucker Carlson", "Oprah Winfrey", "George Clooney", "Meryl Streep", "Alicia Keys", "Beyoncé", "Lady Gaga", "Katy Perry"],
  "Adin Ross": ["MrBeast", "Elon Musk", "Joe Rogan", "Tucker Carlson", "Charlie Kirk", "Jake Paul", "Logan Paul", "Kai Cenat", "IShowSpeed", "Lil Pump", "Donald Trump", "Candace Owens", "Sean Hannity", "Laura Loomer", "Scotty Kilmer", "Tek Syndicate", "Jensen Huang", "Gordon Mah Ung", "Richard Heart", "Kelsey Grammer"],
  "Alec Baldwin": ["Hillary Clinton", "Barack Obama", "Meryl Streep", "Robert De Niro", "George Clooney", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Kerry Washington", "Steven Spielberg", "Julia Roberts", "Matt Damon", "Brad Pitt", "Arnold Schwarzenegger"],
  "Alex Morgan": ["Serena Williams", "Beyoncé", "LeBron James", "Oprah Winfrey", "Hillary Clinton", "Kamala Harris", "Mia Hamm", "Carli Lloyd", "Abby Wambach", "Megan Rapinoe", "George Clooney", "Alicia Keys", "John Legend", "Lady Gaga", "Katy Perry", "Taylor Swift", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin"],
  "Alex Rodriguez": ["Aaron Judge", "Derek Jeter", "Donald Trump", "LeBron James", "Arnold Schwarzenegger", "Jennifer Lopez", "CC Sabathia", "Mariano Rivera", "David Ortiz", "Pedro Martinez", "Joe Rogan", "Tucker Carlson", "Oprah Winfrey", "George Clooney", "Beyoncé", "Alicia Keys", "Lady Gaga", "Katy Perry", "Meryl Streep", "Eva Longoria"],
  "Alicia Keys": ["Hillary Clinton", "Barack Obama", "Michelle Obama", "Joe Biden", "Kamala Harris", "Beyoncé", "Oprah Winfrey", "John Legend", "Lady Gaga", "Katy Perry", "Demi Lovato", "Taylor Swift", "George Clooney", "LeBron James", "Eva Longoria", "America Ferrera", "Ariana Grande", "Meryl Streep", "Alec Baldwin", "Kerry Washington"],
  "America Ferrera": ["Hillary Clinton", "Kamala Harris", "Eva Longoria", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "George Clooney", "Meryl Streep", "Alec Baldwin", "Kerry Washington", "Steven Spielberg", "Julia Roberts", "Ariana Grande", "Salma Hayek"],
  "Ariana Grande": ["Hillary Clinton", "Beyoncé", "Lady Gaga", "Katy Perry", "Taylor Swift", "Oprah Winfrey", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "George Clooney", "Eva Longoria", "America Ferrera", "Kerry Washington", "Meryl Streep", "Alec Baldwin", "Nicki Minaj", "Miley Cyrus", "Selena Gomez", "Justin Bieber"],
  "Arnold Schwarzenegger": ["Mel Gibson", "Sylvester Stallone", "Jon Voight", "Donald Trump", "Tucker Carlson", "Danny Glover", "Chris Pine", "Meryl Streep", "Julia Roberts", "Barack Obama", "Joe Biden", "Kamala Harris", "Joe Rogan", "LeBron James", "Oprah Winfrey", "Linda Hamilton", "James Cameron", "Lou Ferrigno", "Dwayne Johnson", "Kevin Hart", "Maria Shriver"],
  "Barack Obama": ["Hillary Clinton", "Michelle Obama", "Joe Biden", "Kamala Harris", "Beyoncé", "Justin Trudeau", "George Clooney", "Oprah Winfrey", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin", "Jodie Foster", "Julia Roberts", "Arnold Schwarzenegger", "Rothschild"],
  "Beyoncé": ["Hillary Clinton", "Barack Obama", "Michelle Obama", "Joe Biden", "Kamala Harris", "Taylor Swift", "Ariana Grande", "Oprah Winfrey", "LeBron James", "Jay-Z", "Alicia Keys", "John Legend", "Lady Gaga", "Katy Perry", "Demi Lovato", "George Clooney", "Leonardo DiCaprio", "Steven Spielberg", "Eva Longoria", "America Ferrera", "Rothschild", "Alex Morgan"],
  "Bruce Springsteen": ["Hillary Clinton", "Barack Obama", "Joe Biden", "Kamala Harris", "Beyoncé", "Taylor Swift", "Oprah Winfrey", "George Clooney", "Leonardo DiCaprio", "Jon Bon Jovi", "Katy Perry", "Lady Gaga", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Kerry Washington", "Alec Baldwin"],
  "Caitlyn Jenner": ["Donald Trump", "Kid Rock", "Hulk Hogan", "Jon Voight", "Dennis Quaid", "Roseanne Barr", "Scott Baio", "Kelsey Grammer", "Sylvester Stallone", "Mel Gibson", "Jason Aldean", "Amber Rose", "Savannah Chrisley", "Elon Musk", "Joe Rogan", "Tucker Carlson", "Charlie Kirk", "Candace Owens", "Laura Loomer", "Sean Hannity"],
  "Chadwick Boseman": ["Robert Downey Jr.", "Jennifer Lawrence", "Michael B. Jordan", "Ryan Coogler", "Denzel Washington", "Oprah Winfrey", "Lupita Nyong'o", "Danai Gurira", "Hillary Clinton", "Barack Obama", "LeBron James", "George Clooney", "Meryl Streep", "Alicia Keys", "Beyoncé", "John Legend", "Lady Gaga", "Katy Perry", "Eva Longoria", "America Ferrera"],
  "Charli D'Amelio": ["Kim Kardashian", "MrBeast", "Adin Ross", "Addison Rae", "Dixie D'Amelio", "Ariana Grande", "Taylor Swift", "Paris Hilton", "Elon Musk", "Joe Rogan", "Oprah Winfrey", "Beyoncé", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep"],
  "Chris Pine": ["Mel Gibson", "Nick Stahl", "Robert Downey Jr.", "Gal Gadot", "Chris Hemsworth", "Zoe Saldana", "Karl Urban", "Benedict Cumberbatch", "Sylvester Stallone", "Jon Voight", "Donald Trump", "Tucker Carlson", "Sean Hannity", "Charlie Kirk", "Candace Owens", "Hulk Hogan", "Kid Rock", "Roseanne Barr", "Scott Baio", "Kelsey Grammer", "Arnold Schwarzenegger", "Daniel Craig"],
  "Chris Tucker": ["Jackie Chan", "Mel Gibson", "Robert Downey Jr.", "Will Smith", "Eddie Murphy", "Ice Cube", "Donald Trump", "Joe Rogan", "Tucker Carlson", "Oprah Winfrey", "George Clooney", "Meryl Streep", "Alicia Keys", "Beyoncé", "John Legend", "Lady Gaga", "Katy Perry", "LeBron James", "Eva Longoria", "America Ferrera"],
  "Conan O'Brien": ["Will Ferrell", "Adam Sandler", "Jon Stewart", "John Oliver", "Jimmy Fallon", "Jimmy Kimmel", "Tina Fey", "Amy Poehler", "Hillary Clinton", "Barack Obama", "Oprah Winfrey", "George Clooney", "Meryl Streep", "Alicia Keys", "Beyoncé", "John Legend", "Lady Gaga", "Katy Perry", "LeBron James", "Eva Longoria"],
  "Daniel Craig": ["Chris Pine", "Jodie Foster", "Meryl Streep", "Ralph Fiennes", "Léa Seydoux", "Christopher Nolan", "George Clooney", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Le的多", "Eva Longoria", "America Ferrera", "Arnold Schwarzenegger", "Mel Gibson", "Robert Downey Jr."],
  "Danny Glover": ["Mel Gibson", "Barack Obama", "Harry Belafonte", "Hillary Clinton", "Joe Biden", "Kamala Harris", "Alicia Keys", "John Legend", "Oprah Winfrey", "LeBron James", "George Clooney", "Meryl Streep", "Alec Baldwin", "Julia Roberts", "Denzel Washington", "Morgan Freeman", "Samuel L. Jackson", "Forest Whitaker", "Sidney Poitier", "Will Smith", "Arnold Schwarzenegger"],
  "Demi Lovato": ["Hillary Clinton", "Barack Obama", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "Ariana Grande", "Oprah Winfrey", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Kerry Washington", "Meryl Streep", "Alec Baldwin", "Selena Gomez", "Miley Cyrus", "Nicki Minaj", "Christina Aguilera"],
  "Derek Jeter": ["Aaron Judge", "Alex Rodriguez", "Donald Trump", "LeBron James", "Arnold Schwarzenegger", "CC Sabathia", "Mariano Rivera", "Andy Pettitte", "Jorge Posada", "Joe Rogan", "Tucker Carlson", "Oprah Winfrey", "George Clooney", "Beyoncé", "Alicia Keys", "Lady Gaga", "Katy Perry", "Meryl Streep", "Eva Longoria", "America Ferrera"],
  "Eddie Redmayne": ["Emma Watson", "J.K. Rowling", "Daniel Radcliffe", "Rupert Grint", "Meryl Streep", "George Clooney", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Jodie Foster", "Julia Roberts", "Arnold Schwarzenegger", "Chris Pine"],
  "Elon Musk": ["Jensen Huang", "Mark Zuckerberg", "Bill Gates", "Joe Rogan", "Tucker Carlson", "MrBeast", "Kim Kardashian", "Scotty Kilmer", "Tek Syndicate", "Gordon Mah Ung", "Richard Heart", "Caitlyn Jenner", "Donald Trump", "Kid Rock", "Hulk Hogan", "Jon Voight", "Dennis Quaid", "Roseanne Barr", "Scott Baio", "Kelsey Grammer", "Sylvester Stallone", "Jason Aldean"],
  "Emma Watson": ["J.K. Rowling", "Eddie Redmayne", "Hillary Clinton", "Taylor Swift", "Daniel Radcliffe", "Rupert Grint", "Meryl Streep", "George Clooney", "Oprah Winfrey", "Beyoncé", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Alec Baldwin", "Jodie Foster"],
  "Eva Longoria": ["Hillary Clinton", "Kamala Harris", "America Ferrera", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "George Clooney", "Meryl Streep", "Alec Baldwin", "Kerry Washington", "Salma Hayek", "Jennifer Lopez", "Jessica Alba", "Rosario Dawson"],
  "George Clooney": ["Hillary Clinton", "Barack Obama", "Joe Biden", "Amal Clooney", "Beyoncé", "Taylor Swift", "Oprah Winfrey", "Leonardo DiCaprio", "Steven Spielberg", "Matt Damon", "Brad Pitt", "Julia Roberts", "Meryl Streep", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Arnold Schwarzenegger"],
  "Gordon Mah Ung": ["Jensen Huang", "Elon Musk", "Mark Zuckerberg", "Bill Gates", "Linus Tech Tips", "Doug DeMuro", "Scotty Kilmer", "Lisa Su", "Satya Nadella", "Tim Cook", "Sundar Pichai", "Walt Mossberg", "Kara Swisher", "Joe Rogan", "Jay Leno", "Tucker Carlson", "Sean Hannity", "Charlie Kirk", "Anand Shimpi", "Ryan Shrout", "Gordon Moore"],
  "Harrison Ford": ["Mark Hamill", "Carrie Fisher", "George Lucas", "Chris Pine", "Daniel Craig", "Meryl Streep", "George Clooney", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Arnold Schwarzenegger", "Mel Gibson", "Robert Downey Jr."],
  "Hasan Piker": ["Joe Rogan", "Rachel Maddow", "Trevor Noah", "Bernie Sanders", "Alexandria Ocasio-Cortez", "Cenk Uygur", "Hillary Clinton", "Barack Obama", "Kamala Harris", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep"],
  "Heath Ledger": ["Mel Gibson", "Nicole Kidman", "Russell Crowe", "Cate Blanchett", "Chris Pine", "Jake Gyllenhaal", "Christian Bale", "Maggie Gyllenhaal", "Naomi Watts", "Hugh Jackman", "Baz Luhrmann", "Joel Edgerton", "Guy Pearce", "Rose Byrne", "Toni Collette", "Rachel Griffiths", "Eric Bana", "Ben Mendelsohn", "Abbie Cornish", "Mia Wasikowska"],
  "Hillary Clinton": ["Barack Obama", "Michelle Obama", "Joe Biden", "Kamala Harris", "Bill Clinton", "Bernie Sanders", "Elizabeth Warren", "Donald Trump", "Chelsea Clinton", "John Podesta", "Beyoncé", "Taylor Swift", "Oprah Winfrey", "George Clooney", "Lady Gaga", "Katy Perry", "LeBron James", "John Legend", "Alicia Keys", "Demi Lovato", "Rothschild"],
  "Hulk Hogan": ["Donald Trump", "Kid Rock", "Jon Voight", "Dennis Quaid", "Roseanne Barr", "Scott Baio", "Kelsey Grammer", "Sylvester Stallone", "Mel Gibson", "Jason Aldean", "Amber Rose", "Savannah Chrisley", "Elon Musk", "Joe Rogan", "Tucker Carlson", "Charlie Kirk", "Candace Owens", "Laura Loomer", "Sean Hannity", "Roger Stone"],
  "Ivanka Trump": ["Donald Trump", "Jared Kushner", "Melania Trump", "Donald Trump Jr.", "Eric Trump", "Tucker Carlson", "Sean Hannity", "Roger Stone", "Laura Loomer", "Charlie Kirk", "Ghislaine Maxwell", "Jeffrey Epstein", "Paris Hilton", "Karlie Kloss", "Kellyanne Conway", "Steve Bannon", "Mike Pence", "Rupert Murdoch", "Arianna Huffington", "Dasha Zhukova", "Ivana Trump", "Tiffany Trump", "Jensen Huang", "Rothschild"],
  "J.K. Rowling": ["Emma Watson", "Eddie Redmayne", "Daniel Radcliffe", "Rupert Grint", "Meryl Streep", "Oprah Winfrey", "Hillary Clinton", "Barack Obama", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "George Clooney", "Alec Baldwin", "Jodie Foster"],
  "Jackie Chan": ["Chris Tucker", "Mel Gibson", "Arnold Schwarzenegger", "Will Smith", "Jet Li", "Donald Trump", "Joe Rogan", "Tucker Carlson", "Oprah Winfrey", "George Clooney", "Meryl Streep", "Alicia Keys", "Beyoncé", "John Legend", "Lady Gaga", "Katy Perry", "LeBron James", "Eva Longoria", "America Ferrera", "Robert Downey Jr."],
  "Jason Aldean": ["Donald Trump", "Kid Rock", "Hulk Hogan", "Jon Voight", "Dennis Quaid", "Roseanne Barr", "Scott Baio", "Kelsey Grammer", "Sylvester Stallone", "Mel Gibson", "Amber Rose", "Savannah Chrisley", "Elon Musk", "Joe Rogan", "Tucker Carlson", "Charlie Kirk", "Candace Owens", "Laura Loomer", "Sean Hannity", "Roger Stone"],
  "Jeffrey Epstein": ["Ghislaine Maxwell", "Bill Clinton", "Donald Trump", "Alan Dershowitz", "Prince Andrew", "Bill Gates", "David Icke", "Ivanka Trump", "Leonardo DiCaprio", "Mel Gibson", "Chris Pine", "Nick Stahl", "Rothschild"],
  "Jennifer Lawrence": ["Robert Downey Jr.", "Chadwick Boseman", "Meryl Streep", "George Clooney", "Hillary Clinton", "Amy Schumer", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Alec Baldwin", "Jodie Foster", "Julia Roberts"],
  "Jensen Huang": ["Elon Musk", "Mark Zuckerberg", "Bill Gates", "Lisa Su", "Satya Nadella", "Tim Cook", "Sundar Pichai", "Andy Jassy", "Warren Buffett", "Larry Ellison", "Joe Rogan", "Leonardo DiCaprio", "Oprah Winfrey", "Tucker Carlson", "Donald Trump", "Ivanka Trump", "Jared Kushner", "Chris Malachowsky", "Curtis Priem", "Marc Benioff", "Ginni Rometty"],
  "Jodie Foster": ["Mel Gibson", "Hillary Clinton", "Meryl Streep", "Robert De Niro", "Alec Baldwin", "George Clooney", "Julia Roberts", "Matt Damon", "Brad Pitt", "Steven Spielberg", "Leonardo DiCaprio", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria"],
  "John Legend": ["Hillary Clinton", "Barack Obama", "Michelle Obama", "Joe Biden", "Kamala Harris", "Beyoncé", "Alicia Keys", "Oprah Winfrey", "Taylor Swift", "Lady Gaga", "Katy Perry", "Demi Lovato", "LeBron James", "George Clooney", "Eva Longoria", "America Ferrera", "Kerry Washington", "Alec Baldwin", "Meryl Streep", "Chrissy Teigen"],
  "John Oliver": ["Jon Stewart", "Rachel Maddow", "Trevor Noah", "Conan O'Brien", "Stephen Colbert", "Hillary Clinton", "Barack Obama", "Kamala Harris", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin"],
  "John Podesta": ["Hillary Clinton", "Bill Clinton", "Barack Obama", "Joe Biden", "Kamala Harris", "Bernie Sanders", "Elizabeth Warren", "Chelsea Clinton", "Susan Rice", "Valerie Jarrett", "David Plouffe", "David Axelrod", "Neera Tanden", "Tom Daschle", "James Carville", "Paul Begala", "Tony Podesta", "Al Gore", "John Kerry", "George Soros", "Donna Brazile"],
  "Jon Stewart": ["John Oliver", "Rachel Maddow", "Trevor Noah", "Conan O'Brien", "Stephen Colbert", "Hillary Clinton", "Barack Obama", "Kamala Harris", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin"],
  "Justin Trudeau": ["Barack Obama", "Hillary Clinton", "Kamala Harris", "Joe Biden", "Oprah Winfrey", "George Clooney", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin", "Jodie Foster", "Julia Roberts", "Arnold Schwarzenegger"],
  "Katy Perry": ["Hillary Clinton", "Barack Obama", "Beyoncé", "Taylor Swift", "Lady Gaga", "Ariana Grande", "Oprah Winfrey", "George Clooney", "Leonardo DiCaprio", "Steven Spielberg", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Kerry Washington", "Alec Baldwin", "Meryl Streep", "Julia Roberts", "Orlando Bloom"],
  "Keith Richards": ["Paul McCartney", "Mick Jagger", "Willie Nelson", "Elton John", "Bruce Springsteen", "Oprah Winfrey", "George Clooney", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin", "Jodie Foster", "Julia Roberts"],
  "Kelsey Grammer": ["Donald Trump", "Jon Voight", "Sylvester Stallone", "Mel Gibson", "Kid Rock", "Hulk Hogan", "Roseanne Barr", "Scott Baio", "Dennis Quaid", "Jason Aldean", "Amber Rose", "Savannah Chrisley", "Elon Musk", "Joe Rogan", "Tucker Carlson", "Charlie Kirk", "Candace Owens", "Laura Loomer", "Sean Hannity", "Roger Stone"],
  "Kelsey Mitchell": ["Caitlin Clark", "Sabrina Ionescu", "LeBron James", "Alex Morgan", "A'ja Wilson", "Breanna Stewart", "Hillary Clinton", "Kamala Harris", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin", "George Clooney"],
  "Kim Kardashian": ["Charli D'Amelio", "Caitlin Clark", "Tyra Banks", "Elon Musk", "Kanye West", "Paris Hilton", "Kris Jenner", "Kourtney Kardashian", "Khloé Kardashian", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep"],
  "Lady Gaga": ["Hillary Clinton", "Barack Obama", "Beyoncé", "Taylor Swift", "Katy Perry", "Ariana Grande", "Oprah Winfrey", "George Clooney", "Leonardo DiCaprio", "Steven Spielberg", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Kerry Washington", "Alec Baldwin", "Meryl Streep", "Julia Roberts"],
  "LeBron James": ["Hillary Clinton", "Barack Obama", "Joe Biden", "Kamala Harris", "Beyoncé", "Taylor Swift", "Oprah Winfrey", "George Clooney", "Leonardo DiCaprio", "Steven Spielberg", "John Legend", "Alicia Keys", "Lady Gaga", "Katy Perry", "Demi Lovato", "Eva Longoria", "America Ferrera", "Kerry Washington", "Alec Baldwin", "Meryl Streep", "Arnold Schwarzenegger"],
  "Mark Hamill": ["Harrison Ford", "Carrie Fisher", "George Lucas", "Meryl Streep", "George Clooney", "Hillary Clinton", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Alec Baldwin", "Jodie Foster", "Julia Roberts", "Arnold Schwarzenegger"],
  "Mark Zuckerberg": ["Elon Musk", "Jensen Huang", "Bill Gates", "Donald Trump", "Ivanka Trump", "Jared Kushner", "Gordon Mah Ung", "Richard Heart", "Tucker Carlson", "Joe Rogan", "Sean Hannity", "Charlie Kirk", "Candace Owens", "Laura Loomer", "Kid Rock", "Hulk Hogan", "Jon Voight", "Dennis Quaid", "Roseanne Barr", "Scott Baio"],
  "Mel Gibson": ["Donald Trump", "Jon Voight", "Sylvester Stallone", "Kid Rock", "Hulk Hogan", "Roseanne Barr", "Scott Baio", "Danny Glover", "Julia Roberts", "Nick Stahl", "Chris Pine", "Jodie Foster", "Heath Ledger", "Robert Downey Jr.", "Russell Crowe", "Nicole Kidman", "Cate Blanchett", "Jim Caviezel", "Sean Penn", "Joe Rogan", "Tucker Carlson", "Arnold Schwarzenegger"],
  "Meryl Streep": ["Hillary Clinton", "Barack Obama", "Joe Biden", "George Clooney", "Alec Baldwin", "Robert De Niro", "Julia Roberts", "Jodie Foster", "Matt Damon", "Brad Pitt", "Steven Spielberg", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "Arnold Schwarzenegger"],
  "Michael B. Jordan": ["Ryan Coogler", "Chadwick Boseman", "Denzel Washington", "Oprah Winfrey", "Lupita Nyong'o", "Hillary Clinton", "Barack Obama", "LeBron James", "George Clooney", "Meryl Streep", "Alicia Keys", "Beyoncé", "John Legend", "Lady Gaga", "Katy Perry", "Taylor Swift", "Eva Longoria", "America Ferrera", "Jodie Foster", "Julia Roberts"],
  "Mick Jagger": ["Elton John", "Keith Richards", "Willie Nelson", "Bruce Springsteen", "Oprah Winfrey", "George Clooney", "Beyoncé", "Taylor Swift", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera", "Meryl Streep", "Alec Baldwin", "Jodie Foster", "Julia Roberts", "Arnold Schwarzenegger"],
  "MrBeast": ["Adin Ross", "Elon Musk", "Scotty Kilmer", "Tek Syndicate", "Casey Neistat", "Joe Rogan", "Tucker Carlson", "Jensen Huang", "Gordon Mah Ung", "Numberphile", "Veritasium", "ChrisFix", "Tavarish", "Adam LZ", "Jon Rettinger", "Austin Evans", "Jonathan Morrison", "Marques Brownlee", "Jay Leno", "Charlie Kirk"],
  "Neil deGrasse Tyson": ["Stephen Hawking", "Numberphile", "Veritasium", "Brian Cox", "Richard Dawkins", "Joe Rogan", "Elon Musk", "Jensen Huang", "Gordon Mah Ung", "Scotty Kilmer", "Tek Syndicate", "MrBeast", "Casey Neistat", "ChrisFix", "Tavarish", "Adam LZ", "Jon Rettinger", "Austin Evans", "Jonathan Morrison", "Marques Brownlee"],
  "Nick Stahl": ["Mel Gibson", "Chris Pine", "Donald Trump", "Jon Voight", "Sylvester Stallone", "Kid Rock", "Hulk Hogan", "Roseanne Barr", "Scott Baio", "Kelsey Grammer", "Jason Aldean", "Tucker Carlson", "Sean Hannity", "Charlie Kirk", "Candace Owens", "Laura Loomer", "Dennis Quaid", "Robert Downey Jr.", "Jodie Foster", "Russell Crowe"],
  "Nicole Kidman": ["Mel Gibson", "Heath Ledger", "Russell Crowe", "Cate Blanchett", "Hugh Jackman", "Naomi Watts", "Baz Luhrmann", "Joel Edgerton", "Guy Pearce", "Rose Byrne", "Toni Collette", "Rachel Griffiths", "Eric Bana", "Ben Mendelsohn", "Abbie Cornish", "Mia Wasikowska", "Chris Pine", "Jodie Foster", "Meryl Streep", "Tom Cruise"],
  "Numberphile": ["Veritasium", "Neil deGrasse Tyson", "Scotty Kilmer", "Tek Syndicate", "Hank Green", "Brady Haran", "Joe Rogan", "Elon Musk", "Jensen Huang", "Gordon Mah Ung", "MrBeast", "Casey Neistat", "ChrisFix", "Tavarish", "Adam LZ", "Jon Rettinger", "Austin Evans", "Jonathan Morrison", "Marques Brownlee", "Jay Leno"],
  "Oprah Winfrey": ["Hillary Clinton", "Barack Obama", "Michelle Obama", "Joe Biden", "Kamala Harris", "Beyoncé", "Taylor Swift", "George Clooney", "Leonardo DiCaprio", "Steven Spielberg", "Lady Gaga", "Katy Perry", "John Legend", "Alicia Keys", "Demi Lovato", "LeBron James", "Eva Longoria", "America Ferrera", "Kerry Washington", "Alec Baldwin", "Jensen Huang", "Rothschild", "Arnold Schwarzenegger"],
  "Orlando Bloom": ["Katy Perry", "Miranda Kerr", "Leonardo DiCaprio", "Meryl Streep", "George Clooney", "Oprah Winfrey", "Beyoncé", "Taylor Swift", "Lady Gaga", "John Legend", "Alicia Keys", "LeBron James", "Eva Longoria", "America Ferrera",]
}
    
    
    # Load existing JSON
    data = load_json(input_path)
    
    # Inject new entries and sort
    updated_data = inject_entries(data, new_entries)
    
    # Save or print updated JSON
    save_json(updated_data, output_path)
    print(f"Updated JSON saved to {output_path}")
    # Optional: Print to console
    # print(json.dumps(updated_data, indent=2, sort_keys=True))

if __name__ == "__main__":
    main()