import re
from pathlib import Path

# Root folder of your website files
ROOT = Path("/home/josefkulovany")

# Folder(s) where local images reside (adjust if images scattered elsewhere)
LOCAL_IMAGE_FOLDERS = [
    ROOT / "images",
    ROOT / "assets" / "images",
    # add more if needed
]

# Regex to find remote Zyro CDN URLs in img src or other tags
REMOTE_IMG_URL_PATTERN = re.compile(
    r'https://assets\.zyrosite\.com/[^"\']+'
)

def find_local_image(filename):
    """Search all LOCAL_IMAGE_FOLDERS for a file with given filename"""
    for folder in LOCAL_IMAGE_FOLDERS:
        candidate = folder / filename
        if candidate.exists():
            return candidate.relative_to(ROOT)
    # fallback: try anywhere under ROOT
    for candidate in ROOT.rglob(filename):
        if candidate.is_file():
            return candidate.relative_to(ROOT)
    return None

def fix_file(file_path):
    content = file_path.read_text(encoding="utf-8", errors="ignore")
    changed = False
    urls = set(REMOTE_IMG_URL_PATTERN.findall(content))

    for url in urls:
        filename = url.split("/")[-1].split("?")[0].split("#")[0]
        local_rel_path = find_local_image(filename)
        if local_rel_path:
            # Use forward slashes in HTML paths
            local_rel_str = "/" + str(local_rel_path).replace("\\", "/")
            content = content.replace(url, local_rel_str)
            changed = True
            print(f"Replaced {url} → {local_rel_str} in {file_path}")
        else:
            print(f"⚠️ WARNING: local image file not found for {filename} referenced in {file_path}")

    if changed:
        file_path.write_text(content, encoding="utf-8")

def main():
    print(f"Scanning {ROOT} for .php and .html files to fix image URLs...\n")
    for file_path in ROOT.rglob("*"):
        if file_path.suffix.lower() in {".php", ".html"}:
            fix_file(file_path)
    print("\nAll done!")

if __name__ == "__main__":
    main()
