import os
from pathlib import Path
from urllib.parse import urljoin, quote, urlparse
import base64
import mimetypes
import requests
from bs4 import BeautifulSoup

SCRIPT_DIR = Path(__file__).resolve().parent
INPUT_DIR = SCRIPT_DIR
OUTPUT_DIR = SCRIPT_DIR / "recompiled-pages"
BASE_URL = "https://josefkulovany.com/"

def sanitize_filename(filename):
    return quote(filename, safe='').replace('%20', '_').replace('%', '_')

def encode_to_base64(local_path):
    try:
        mime, _ = mimetypes.guess_type(str(local_path))
        mime = mime or "application/octet-stream"
        with open(local_path, "rb") as f:
            encoded = base64.b64encode(f.read()).decode('utf-8')
        return f"data:{mime};base64,{encoded}"
    except Exception:
        return None

def fetch_remote_content(url):
    try:
        resp = requests.get(url, timeout=10)
        if resp.status_code == 200:
            return resp.text
    except:
        return None

def inline_assets(soup, base_path):
    # Inline CSS
    for link in soup.find_all("link", rel="stylesheet"):
        href = link.get("href")
        if not href:
            continue
        if href.startswith("http") or href.startswith("//"):
            content = fetch_remote_content(href)
        else:
            local_path = (base_path / href).resolve()
            if local_path.exists():
                with open(local_path, "r", encoding="utf-8") as f:
                    content = f.read()
            else:
                content = None
        if content:
            style_tag = soup.new_tag("style")
            style_tag.string = content
            link.replace_with(style_tag)

    # Inline JS
    for script in soup.find_all("script", src=True):
        src = script["src"]
        if src.startswith("http") or src.startswith("//"):
            content = fetch_remote_content(src)
        else:
            local_path = (base_path / src).resolve()
            if local_path.exists():
                with open(local_path, "r", encoding="utf-8") as f:
                    content = f.read()
            else:
                content = None
        if content:
            new_script = soup.new_tag("script")
            new_script.string = content
            script.replace_with(new_script)

    # Inline IMG
    for img in soup.find_all("img", src=True):
        src = img["src"]
        local_path = (base_path / src).resolve()
        if local_path.exists():
            data_uri = encode_to_base64(local_path)
            if data_uri:
                img["src"] = data_uri
                img.attrs.pop("srcset", None)
        else:
            img["src"] = urljoin(BASE_URL, src)

    # Fix iframes (absolute URLs)
    for iframe in soup.find_all("iframe", src=True):
        iframe["src"] = urljoin(BASE_URL, iframe["src"])

    return soup

def fix_head_masthead(soup):
    if not soup.find("base"):
        soup.head.insert(0, soup.new_tag("base", href="./"))

    # Optional: add custom masthead CSS if desired
    # style_tag = soup.new_tag("style")
    # style_tag.string = "/* custom masthead fixes */"
    # soup.head.append(style_tag)

    return soup

def process_file(file_path):
    base_path = file_path.parent
    with open(file_path, "r", encoding="utf-8") as f:
        soup = BeautifulSoup(f, "html.parser")

    soup = fix_head_masthead(soup)
    soup = inline_assets(soup, base_path)
    return str(soup)

def recompile_all():
    for file_path in INPUT_DIR.rglob("*.html"):
        relative_path = file_path.relative_to(INPUT_DIR)
        output_file = OUTPUT_DIR / relative_path
        output_file.parent.mkdir(parents=True, exist_ok=True)
        try:
            result = process_file(file_path)
            with open(output_file, "w", encoding="utf-8") as out:
                out.write(result)
            print(f"✔️ Recompiled: {relative_path}")
        except Exception as e:
            print(f"❌ Failed: {relative_path} — {e}")

if __name__ == "__main__":
    recompile_all()
