import os
from bs4 import BeautifulSoup
from pathlib import Path
import base64
import re
from htmlmin import minify
from csscompressor import compress
from jsmin import jsmin

SRC_DIR = Path("source-html")
OUT_DIR = Path("recompiled-pages")
ASSETS_DIR = Path("assets")
OUT_DIR.mkdir(parents=True, exist_ok=True)

# Load universal lightbox script and css (replace with your preferred one)
LIGHTBOX_JS = (ASSETS_DIR / "lightgallery.min.js").read_text()
LIGHTBOX_CSS = (ASSETS_DIR / "lightgallery.min.css").read_text()

def fix_links(soup):
    for a in soup.find_all("a", href=True):
        href = a['href']
        if "josefkulovany.com/" in href:
            target = href.split("josefkulovany.com/")[-1].rstrip('/')
            target = target.replace(" ", "%20")  # normalize
            if not target.endswith(".html"):
                new_href = f"recompiled-pages/{target.capitalize()}.html"
                a['href'] = new_href
    return soup

def inline_css_js(soup, local_path):
    # Inline styles
    for link in soup.find_all("link", rel="stylesheet", href=True):
        href = link['href']
        if href.startswith("http"):
            continue
        try:
            css_file = (local_path.parent / href).resolve()
            css_content = compress(css_file.read_text())
            style_tag = soup.new_tag("style")
            style_tag.string = css_content
            link.replace_with(style_tag)
        except Exception:
            continue

    # Inline scripts
    for script in soup.find_all("script", src=True):
        src = script['src']
        if src.startswith("http"):
            continue
        try:
            js_file = (local_path.parent / src).resolve()
            js_content = jsmin(js_file.read_text())
            new_script = soup.new_tag("script")
            new_script.string = js_content
            script.replace_with(new_script)
        except Exception:
            continue

    return soup

def fix_broken_images(soup):
    for img in soup.find_all("img"):
        src = img.get("src", "")
        if src.startswith("data:image") and len(src) < 100:
            img['src'] = "/assets/broken.png"  # use placeholder or remove
    return soup

def inject_lightbox(soup):
    # Inject CSS and JS for lightbox
    head = soup.head
    style = soup.new_tag("style")
    style.string = LIGHTBOX_CSS
    script = soup.new_tag("script")
    script.string = LIGHTBOX_JS
    head.append(style)
    head.append(script)

    # Add lightbox attributes
    for img in soup.find_all("img"):
        parent = img.find_parent("a")
        if not parent:
            a = soup.new_tag("a", href=img['src'])
            img.wrap(a)
            a['data-lg-size'] = f"{img.get('width', '800')}-{img.get('height', '600')}"
            a['data-lg-image'] = img['src']
    body = soup.body
    init_script = soup.new_tag("script")
    init_script.string = """
        lightGallery(document.body, {
            selector: 'a[data-lg-image]',
            download: false
        });
    """
    body.append(init_script)
    return soup

def process_file(file):
    content = file.read_text()
    soup = BeautifulSoup(content, "html.parser")
    soup = inline_css_js(soup, file)
    soup = fix_links(soup)
    soup = fix_broken_images(soup)
    soup = inject_lightbox(soup)
    html = minify(str(soup), remove_empty_space=True)
    out_path = OUT_DIR / file.name
    out_path.write_text(html)
    print(f"✓ {file.name} written")

for file in SRC_DIR.glob("*.html"):
    process_file(file)
