import os
from pathlib import Path
from urllib.parse import urljoin, quote
import base64
import mimetypes
import requests
from bs4 import BeautifulSoup

SCRIPT_DIR = Path(__file__).resolve().parent
INPUT_DIR = SCRIPT_DIR
OUTPUT_DIR = SCRIPT_DIR / "recompiled-pages"
ASSETS_DIR = SCRIPT_DIR / "assets"
BASE_URL = "https://josefkulovany.com/"

def sanitize_filename(filename):
    return quote(filename, safe='').replace('%20', '_').replace('%', '_')

def encode_to_base64(local_path):
    try:
        mime, _ = mimetypes.guess_type(str(local_path))
        mime = mime or "application/octet-stream"
        with open(local_path, "rb") as f:
            encoded = base64.b64encode(f.read()).decode('utf-8')
        return f"data:{mime};base64,{encoded}"
    except Exception:
        return None

def fetch_remote_content(url):
    try:
        resp = requests.get(url, timeout=10)
        if resp.status_code == 200:
            return resp.text
    except:
        return None

def try_inline_asset(soup, tag, attr_name, wrap_tag, is_binary=False):
    for el in soup.find_all(tag):
        src = el.get(attr_name)
        if not src:
            continue
        content = None
        src_path = Path(src.strip("/"))
        local_path = ASSETS_DIR / src_path

        if src.startswith("http") or src.startswith("//"):
            content = fetch_remote_content(src)
        elif local_path.exists():
            if is_binary:
                data_uri = encode_to_base64(local_path)
                if data_uri:
                    el[attr_name] = data_uri
                    el.attrs.pop("srcset", None)
            else:
                with open(local_path, "r", encoding="utf-8") as f:
                    content = f.read()
        else:
            el[attr_name] = urljoin(BASE_URL, src)

        if content and not is_binary:
            new_tag = soup.new_tag(wrap_tag)
            new_tag.string = content
            el.replace_with(new_tag)

def inline_assets(soup):
    try_inline_asset(soup, "link", "href", "style", is_binary=False)
    try_inline_asset(soup, "script", "src", "script", is_binary=False)
    try_inline_asset(soup, "img", "src", "img", is_binary=True)

    for iframe in soup.find_all("iframe", src=True):
        iframe["src"] = urljoin(BASE_URL, iframe["src"])

    return soup

def fix_head_and_masthead(soup):
    if soup.head and not soup.head.find("base"):
        soup.head.insert(0, soup.new_tag("base", href="./"))

    return soup

def process_file(file_path):
    with open(file_path, "r", encoding="utf-8") as f:
        soup = BeautifulSoup(f, "html.parser")

    soup = fix_head_and_masthead(soup)
    soup = inline_assets(soup)
    return str(soup)

def recompile_all():
    OUTPUT_DIR.mkdir(parents=True, exist_ok=True)

    for file_path in INPUT_DIR.rglob("*.html"):
        if OUTPUT_DIR in file_path.parents:
            continue  # Skip previously compiled output

        relative_path = file_path.relative_to(INPUT_DIR)
        output_file = OUTPUT_DIR / relative_path
        output_file.parent.mkdir(parents=True, exist_ok=True)

        try:
            result = process_file(file_path)
            with open(output_file, "w", encoding="utf-8") as out:
                out.write(result)
            print(f"✔️ Recompiled: {relative_path}")
        except Exception as e:
            print(f"❌ Failed: {relative_path} — {e}")

if __name__ == "__main__":
    recompile_all()
