import os
import base64
import mimetypes
from pathlib import Path
from urllib.parse import urljoin, quote, unquote
from bs4 import BeautifulSoup
import requests

# === Configuration ===
SCRIPT_DIR = Path(__file__).resolve().parent
INPUT_DIR = SCRIPT_DIR
OUTPUT_DIR = SCRIPT_DIR / "recompiled-pages"
OUTPUT_DIR.mkdir(parents=True, exist_ok=True)

BASE_URL = "https://josefkulovany.com/"

masthead_map = {
    "art": "Art _ Josef Kulovany.html",
    "construction": "Construction _ Josef Kulovany.html",
    "companies": "Companies _ Josef Kulovany.html",
    "extracurricular": "Extracurricular _ Josef Kulovany.html",
    "engineering-design": "Engineering _ Design _ Josef Kulovany.html",
    "art-engineering-economics-design-programming-construction": "Art, Engineering, Economics, Design, Programming, Construction _ Josef Kulovany.html",
    "latest-projects": "Latest Projects _ Josef Kulovany.htm"
}

def sanitize_filename(fn):
    return quote(fn, safe='').replace('%20', '_').replace('%', '_')

def encode_image(path: Path):
    try:
        mime, _ = mimetypes.guess_type(path.name)
        mime = mime or "application/octet-stream"
        with open(path, "rb") as f:
            encoded = base64.b64encode(f.read()).decode()
            return f"data:{mime};base64,{encoded}"
    except:
        return None

def fetch_remote_image(url):
    try:
        r = requests.get(url, timeout=10)
        r.raise_for_status()
        mime, _ = mimetypes.guess_type(url)
        mime = mime or "application/octet-stream"
        encoded = base64.b64encode(r.content).decode()
        return f"data:{mime};base64,{encoded}"
    except:
        return None

def fetch_text(url: str) -> str:
    try:
        r = requests.get(url, timeout=10)
        r.raise_for_status()
        return r.text
    except:
        return None

def process_html(file_path: Path):
    soup = BeautifulSoup(file_path.read_text(encoding='utf-8', errors='ignore'), "html.parser")
    base_path = file_path.parent

    if soup.head and not soup.head.find("base"):
        soup.head.insert(0, soup.new_tag("base", href="./"))

    # Inline styles
    for link in soup.find_all("link", rel="stylesheet", href=True):
        href = link['href']
        content = fetch_text(href) if href.startswith("http") else (base_path / href).read_text(encoding='utf-8', errors='ignore') if (base_path / href).exists() else None
        if content:
            style = soup.new_tag("style")
            style.string = content
            link.replace_with(style)

    # Inline JS
    for script in soup.find_all("script", src=True):
        src = script['src']
        content = fetch_text(src) if src.startswith("http") else (base_path / src).read_text(encoding='utf-8', errors='ignore') if (base_path / src).exists() else None
        if content:
            new = soup.new_tag("script")
            new.string = content
            script.replace_with(new)

    # Fix and wrap images
    for img in soup.find_all("img", src=True):
        src = img['src']
        decoded_src = unquote(src)
        local = None

        if not decoded_src.startswith(("http", "//")):
            if decoded_src.startswith("/"):
                local = (INPUT_DIR / decoded_src.lstrip('/')).resolve()
            else:
                local = (base_path / decoded_src).resolve()

        data_uri = encode_image(local) if local and local.exists() else fetch_remote_image(urljoin(BASE_URL, decoded_src))
        if data_uri:
            img['data-orig-src'] = src
            img['src'] = data_uri
            img.attrs.pop("srcset", None)
        else:
            img['src'] = urljoin(BASE_URL, decoded_src)

        if img.parent.name != "a":
            wrapper = soup.new_tag("a", href=img['data-orig-src'] if "data-orig-src" in img.attrs else img['src'])
            wrapper['class'] = 'zoomable'
            img.wrap(wrapper)

    # Lightbox script
    script_tag = soup.new_tag("script")
    script_tag.string = '''
document.addEventListener("click", function(e) {
  const link = e.target.closest("a.zoomable");
  if (!link) return;
  e.preventDefault();
  let existing = document.getElementById("zoom-overlay");
  if (existing) existing.remove();
  const overlay = document.createElement("div");
  overlay.id = "zoom-overlay";
  overlay.style = `
    position:fixed;top:0;left:0;width:100%;height:100%;
    background:rgba(0,0,0,0.8);
    display:flex;align-items:center;justify-content:center;
    cursor:pointer;
    z-index:10000;
  `;
  const img = document.createElement("img");
  img.src = link.href;
  img.style = "max-width:90%; max-height:90%; box-shadow: 0 0 10px black;";
  overlay.appendChild(img);
  overlay.addEventListener("click", () => overlay.remove());
  document.body.appendChild(overlay);
});
    '''
    (soup.body or soup).append(script_tag)

    # Update masthead links
    for a in soup.find_all("a", href=True):
        href = a['href']
        if href.startswith(BASE_URL):
            path = href[len(BASE_URL):].strip("/").lower()
            path = path.split('?')[0].split('#')[0]
            if path in masthead_map:
                a['href'] = masthead_map[path]

    return str(soup)

def recompile_all():
    for ext in ("*.html", "*.htm", "*.mhtml"):
        for src in INPUT_DIR.rglob(ext):
            if OUTPUT_DIR in src.parents:
                continue
            out_rel = src.relative_to(INPUT_DIR)
            out_path = OUTPUT_DIR / out_rel
            out_path.parent.mkdir(parents=True, exist_ok=True)
            html = process_html(src)
            out_path.write_text(html, encoding='utf-8')
            print("✓", out_rel)

if __name__ == "__main__":
    recompile_all()
