import os
from bs4 import BeautifulSoup
from pathlib import Path
import requests
from urllib.parse import urljoin, urlparse, quote
import base64

# Use the actual location of the script
SCRIPT_DIR = Path(__file__).resolve().parent
INPUT_DIR = SCRIPT_DIR
OUTPUT_DIR = SCRIPT_DIR / "recompiled-pages"
BASE_URL = "https://josefkulovany.com/"
EMBED_RESOURCES = False  # Set to True to inline CSS/images as base64

OUTPUT_DIR.mkdir(parents=True, exist_ok=True)

def sanitize_filename(filename):
    return quote(filename, safe='').replace('%20', '_').replace('%', '_')

def embed_resource(url):
    try:
        res = requests.get(url)
        res.raise_for_status()
        content_type = res.headers.get("Content-Type", "application/octet-stream")
        data = base64.b64encode(res.content).decode('utf-8')
        return f"data:{content_type};base64,{data}"
    except Exception as e:
        print(f"Error embedding {url}: {e}")
        return url

def process_html(file_path):
    with open(file_path, "r", encoding="utf-8") as f:
        soup = BeautifulSoup(f, "html.parser")

    # Adjust base tag
    base_tag = soup.find("base")
    if base_tag:
        base_tag["href"] = "./"
    else:
        soup.head.insert(0, soup.new_tag("base", href="./"))

    # Fix resource links
    for tag, attr in [("link", "href"), ("script", "src"), ("img", "src")]:
        for el in soup.find_all(tag):
            if el.has_attr(attr):
                original_url = el[attr]
                full_url = urljoin(BASE_URL, original_url)
                if EMBED_RESOURCES and tag in ["link", "img"]:
                    if tag == "link" and "stylesheet" in el.get("rel", []):
                        try:
                            css = requests.get(full_url).text
                            style_tag = soup.new_tag("style")
                            style_tag.string = css
                            el.replace_with(style_tag)
                        except:
                            continue
                    elif tag == "img":
                        el[attr] = embed_resource(full_url)
                else:
                    el[attr] = full_url

    return str(soup)

def recompile_all():
    for file in INPUT_DIR.glob("*.html"):
        sanitized = sanitize_filename(file.name)
        output_file = OUTPUT_DIR / sanitized
        try:
            compiled_html = process_html(file)
            with open(output_file, "w", encoding="utf-8") as out:
                out.write(compiled_html)
            print(f"✔️ Recompiled: {file.name} -> {output_file.name}")
        except Exception as e:
            print(f"❌ Error processing {file.name}: {e}")

if __name__ == "__main__":
    recompile_all()
