import os
import re
from bs4 import BeautifulSoup
from pathlib import Path
from htmlmin import minify
from csscompressor import compress
from jsmin import jsmin

SRC_DIR = Path("downloaded-site")  # Folder with original HTML
DST_DIR = Path("recompiled-pages")  # Output directory

DST_DIR.mkdir(parents=True, exist_ok=True)

# New masthead HTML snippet (customize as needed)
NEW_MASTHEAD = '''
<nav class="masthead">
  <a href="Art___Josef_Kulovany.html">Art</a> |
  <a href="Companies___Josef_Kulovany.html">Companies</a> |
  <a href="Construction___Josef_Kulovany.html">Construction</a> |
  <a href="Engineering___Design___Josef_Kulovany.html">Engineering + Design</a> |
  <a href="Extracurricular___Josef_Kulovany.html">Extracurricular</a>
</nav>
'''

# Lightbox CSS and JS (embedded directly)
LIGHTBOX_CSS = compress('''
<style>
.lightbox-overlay {
  position: fixed; top: 0; left: 0;
  width: 100%; height: 100%;
  background: rgba(0,0,0,0.8);
  display: none; justify-content: center; align-items: center;
  z-index: 10000;
}
.lightbox-overlay img {
  max-width: 90%; max-height: 90%;
  box-shadow: 0 0 10px #fff;
}
</style>
''')

LIGHTBOX_JS = jsmin('''
<script>
document.addEventListener("DOMContentLoaded", function() {
  const overlay = document.createElement("div");
  overlay.className = "lightbox-overlay";
  const img = document.createElement("img");
  overlay.appendChild(img);
  document.body.appendChild(overlay);
  overlay.addEventListener("click", () => overlay.style.display = "none");
  document.querySelectorAll("img").forEach(el => {
    el.style.cursor = "zoom-in";
    el.addEventListener("click", e => {
      img.src = el.src;
      overlay.style.display = "flex";
    });
  });
});
</script>
''')

def recompile_file(file_path: Path):
    try:
        html = file_path.read_text(encoding='utf-8')
        soup = BeautifulSoup(html, "html.parser")

        # Replace masthead (assumes it's in a <nav> or identifiable div)
        existing_nav = soup.find("nav") or soup.find("div", class_="masthead")
        if existing_nav:
            existing_nav.replace_with(BeautifulSoup(NEW_MASTHEAD, "html.parser"))

        # Fix all <img> paths to relative filenames if possible
        for img in soup.find_all("img"):
            src = img.get("src", "")
            if src.startswith("https://josefkulovany.com/"):
                img["src"] = src.split("/")[-1]  # Just the filename
            elif src.startswith("/"):
                img["src"] = src.lstrip("/")  # Remove leading slash

        # Remove broken zoom or iframe wrappers
        for a in soup.find_all("a", href=True):
            if a.get("href", "").endswith(('.jpg', '.png', '.webp')):
                if a.img:
                    a.replace_with(a.img)  # flatten <a><img></a> into just <img>

        # Inject Lightbox if not already present
        if soup.body:
            soup.body.append(BeautifulSoup(LIGHTBOX_CSS + LIGHTBOX_JS, "html.parser"))

        # Write minified HTML to destination
        dest_path = DST_DIR / file_path.name
        dest_path.write_text(minify(str(soup), remove_empty_space=True), encoding='utf-8')
        print(f"✔ Recompiled: {file_path.name}")
    except Exception as e:
        print(f"✖ Failed on {file_path.name}: {e}")

def recompile_all():
    html_files = list(SRC_DIR.glob("*.html"))
    for file in html_files:
        recompile_file(file)

if __name__ == "__main__":
    recompile_all()
