import os
from pathlib import Path

# Configure these paths as needed:
PROJECT_ROOT = Path("/home/josefkulovany").resolve()
PUBLIC_ASSETS = PROJECT_ROOT / "public_assets"
FALLBACK_DIR = PROJECT_ROOT / "_missing_assets"

PLACEHOLDER_SIGNATURE = "// placeholder file"

def is_symlink(path):
    return path.is_symlink()

def is_placeholder_file(path):
    try:
        if path.is_file():
            content = path.read_text(errors="ignore").strip()
            return content == PLACEHOLDER_SIGNATURE
        return False
    except Exception:
        return False

def remove_symlinks_and_placeholders(base_path):
    removed_symlinks = 0
    removed_placeholders = 0

    for path in base_path.rglob("*"):
        if path.exists():
            if is_symlink(path):
                try:
                    path.unlink()
                    removed_symlinks += 1
                    print(f"Removed symlink: {path}")
                except Exception as e:
                    print(f"Failed to remove symlink {path}: {e}")
            elif is_placeholder_file(path):
                try:
                    path.unlink()
                    removed_placeholders += 1
                    print(f"Removed placeholder: {path}")
                except Exception as e:
                    print(f"Failed to remove placeholder {path}: {e}")
    return removed_symlinks, removed_placeholders

def remove_fallback_downloads(fallback_dir):
    removed_files = 0
    if fallback_dir.exists():
        for path in fallback_dir.iterdir():
            try:
                if path.is_file():
                    path.unlink()
                    removed_files += 1
                    print(f"Removed fallback file: {path}")
            except Exception as e:
                print(f"Failed to remove fallback file {path}: {e}")
    return removed_files

def main():
    print(f"Starting cleanup in project root: {PROJECT_ROOT}")

    # 1. Remove symlinks and placeholders in project root (including asset folders)
    symlinks_removed, placeholders_removed = remove_symlinks_and_placeholders(PROJECT_ROOT)
    
    # 2. Remove symlinks in public_assets folder (if separate)
    if PUBLIC_ASSETS.exists() and PUBLIC_ASSETS.is_dir():
        public_symlinks_removed, public_placeholders_removed = remove_symlinks_and_placeholders(PUBLIC_ASSETS)
        symlinks_removed += public_symlinks_removed
        placeholders_removed += public_placeholders_removed

    # 3. Remove downloaded fallback files
    fallback_removed = remove_fallback_downloads(FALLBACK_DIR)

    print("\n=== Cleanup Summary ===")
    print(f"Symlinks removed: {symlinks_removed}")
    print(f"Placeholders removed: {placeholders_removed}")
    print(f"Fallback downloaded files removed: {fallback_removed}")

if __name__ == "__main__":
    main()
