import pandas as pd
import matplotlib.pyplot as plt
import numpy as np # Import numpy for potential NaN handling or specific calculations

# --- Configuration for loading the data ---
file_path = "emergent_constants.txt"

# --- Load the data ---
try:
    # Assuming the file is tab-separated as per your original code's to_csv
    df_results = pd.read_csv(file_path, sep="\t")
    print(f"Successfully loaded {len(df_results)} constants from {file_path}")
except FileNotFoundError:
    print(f"Error: The file '{file_path}' was not found. Please ensure it's in the same directory as this script.")
    exit()
except Exception as e:
    print(f"An error occurred while loading the file: {e}")
    exit()

# --- Prepare data for plotting ---
# Ensure 'error' and 'n' columns are numeric.
# Errors or non-numeric values might have been written if inversions failed.
df_results['error'] = pd.to_numeric(df_results['error'], errors='coerce')
df_results['n'] = pd.to_numeric(df_results['n'], errors='coerce')

# Drop rows where 'error' or 'n' might be NaN after conversion (e.g., from failed inversions)
df_results_cleaned = df_results.dropna(subset=['error', 'n'])

if df_results_cleaned.empty:
    print("No valid data available for plotting after cleaning. Check your input file.")
else:
    # Sort for consistent plotting, as in the original code
    df_results_sorted = df_results_cleaned.sort_values("error")

    # --- Regenerate Histogram of Absolute Errors ---
    plt.figure(figsize=(10, 5))
    plt.hist(df_results_sorted['error'], bins=50, color='skyblue', edgecolor='black')
    plt.title('Histogram of Absolute Errors in Symbolic Fit')
    plt.xlabel('Absolute Error')
    plt.ylabel('Count')
    plt.grid(True)
    plt.tight_layout()
    plt.show()

    # --- Regenerate Scatter Plot of Absolute Error vs Symbolic Dimension n ---
    plt.figure(figsize=(10, 5))
    plt.scatter(df_results_sorted['n'], df_results_sorted['error'], alpha=0.5, s=15, c='orange', edgecolors='black')
    plt.title('Absolute Error vs Symbolic Dimension n')
    plt.xlabel('n')
    plt.ylabel('Absolute Error')
    plt.grid(True)
    plt.tight_layout()
    plt.show()

    print("\nGraphs regenerated successfully!")